// Copyright The Monogon Project Authors.
// SPDX-License-Identifier: Apache-2.0

// This tool generates //build/analysis/lib:stdlib_packages.go, which contains a
// set of all Go stdlib packges. This is generated ahead of time in the build
// system as it can be an expensive operation that also depends on the presence
// of a working `go` tool environment, so we want to do this as rarely as
// possible.
package main

import (
	"fmt"
	"os"
	"path/filepath"
	"sort"
	"strings"

	"golang.org/x/tools/go/packages"

	"source.monogon.dev/build/toolbase/gotoolchain"
)

func main() {
	os.Setenv("PATH", filepath.Dir(gotoolchain.Go))

	gocache, err := os.MkdirTemp("/tmp", "gocache")
	if err != nil {
		panic(err)
	}
	defer os.RemoveAll(gocache)

	gopath, err := os.MkdirTemp("/tmp", "gopath")
	if err != nil {
		panic(err)
	}
	defer os.RemoveAll(gopath)

	os.Setenv("GOCACHE", gocache)
	os.Setenv("GOPATH", gopath)

	pkgs, err := packages.Load(nil, "std")
	if err != nil {
		panic(err)
	}
	sort.Slice(pkgs, func(i, j int) bool { return pkgs[i].PkgPath < pkgs[j].PkgPath })

	if len(os.Args) != 2 {
		panic("must be called with output file name")
	}

	out, err := os.Create(os.Args[1])
	if err != nil {
		panic(err)
	}
	defer out.Close()

	fmt.Fprintf(out, "// Code generated by //build/analysis/lib/genstd. DO NOT EDIT.\n")
	fmt.Fprintf(out, "package lib\n\n")
	fmt.Fprintf(out, "// StdlibPackages is a set of all package paths that are part of the Go standard\n")
	fmt.Fprintf(out, "// library.\n")
	fmt.Fprintf(out, "var StdlibPackages = map[string]bool{\n")
	for _, pkg := range pkgs {
		path := pkg.PkgPath
		if strings.Contains(path, "/internal/") {
			continue
		}
		if strings.HasPrefix(path, "vendor/") {
			continue
		}
		fmt.Fprintf(out, "\t%q: true,\n", pkg.PkgPath)
	}
	fmt.Fprintf(out, "}\n")
}
