// Copyright The Monogon Project Authors.
// SPDX-License-Identifier: Apache-2.0

package metrics

import (
	"fmt"
	"io"
	"net/http"

	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/client_golang/prometheus/promhttp"

	"source.monogon.dev/metropolis/node/allocs"
	"source.monogon.dev/osbase/supervisor"
)

// An Exporter is a source of Prometheus metrics. There are two possible kinds of
// exporters:
//
// 1. A binary running under the Metrics service which collects some metrics and
// exposes them on a locally bound TCP port (either started by the Exporter or
// already running as part of Metropolis).
//
// 2. An in-memory Prometheus registry/gatherer for metrics generated by the
// Metropolis core process.
//
// The Metrics Service will forward requests from /metrics/<name> to the
// exporter.
type Exporter struct {
	// Name of the exporter, which becomes part of the metrics URL for this exporter.
	Name string
	// Gatherer, if provided, is a Prometheus registry (or other Gatherer) that will
	// be queried for metrics for this exporter. Exactly one of Gatherer or Port must
	// be set.
	Gatherer prometheus.Gatherer
	// Port on which an exporter is/will be running to which metrics requests will be
	// proxied to. Exactly one of Gatherer or Port must be set.
	Port allocs.Port
	// Executable to run to start the exporter. If empty, no executable will be
	// started.
	Executable string
	// Arguments to start the exporter. The exporter should listen at 127.0.0.1 and
	// the port specified by Port, and serve its metrics on /metrics.
	Arguments []string
	// Path to scrape metrics at. Defaults to /metrics.
	Path string
}

// CoreRegistry is the metrics registry that will be served at /core. All
// prometheus metrics exported by the node core should register here.
var CoreRegistry = prometheus.NewRegistry()

// DefaultExporters are the exporters which we run by default in Metropolis.
var DefaultExporters = []*Exporter{
	{
		Name:     "core",
		Gatherer: CoreRegistry,
	},
	{
		Name:       "node",
		Port:       allocs.PortMetricsNodeListener,
		Executable: "/metrics/bin/node_exporter",
		Arguments: []string{
			"--web.listen-address=127.0.0.1:" + allocs.PortMetricsNodeListener.PortString(),
			"--collector.buddyinfo",
			"--collector.zoneinfo",
			"--collector.tcpstat",
			"--collector.cpu.info",
			"--collector.ethtool",
			"--collector.cpu_vulnerabilities",
			"--collector.ethtool.device-exclude=^(veth.*|sit.*|lo|clusternet)$",
			"--collector.netclass.ignored-devices=^(veth.*)$",
			"--collector.netdev.device-exclude=^(veth.*)$",
			"--collector.filesystem.mount-points-exclude=^/(dev|proc|sys|data/kubernetes/kubelet/pods/.+|tmp/.+|ephemeral/containerd/.+)($|/)",
		},
	},
	{
		Name: "etcd",
		Port: allocs.PortMetricsEtcdListener,
	},
	{
		Name: "kubernetes-scheduler",
		Port: allocs.PortMetricsKubeSchedulerListener,
	},
	{
		Name: "kubernetes-controller-manager",
		Port: allocs.PortMetricsKubeControllerManagerListener,
	},
	{
		Name: "kubernetes-apiserver",
		Port: allocs.PortMetricsKubeAPIServerListener,
	},
	{
		Name: "containerd",
		Port: allocs.PortMetricsContainerdListener,
		Path: "/v1/metrics",
	},
}

func (e *Exporter) serveHTTPForward(w http.ResponseWriter, r *http.Request) {
	ctx := r.Context()

	// We are supplying the http.Server with a BaseContext that contains the
	// context from our runnable which contains the logger.
	logger := supervisor.Logger(ctx)

	path := e.Path
	if e.Path == "" {
		path = "/metrics"
	}

	url := "http://127.0.0.1:" + e.Port.PortString() + path
	outReq, err := http.NewRequestWithContext(ctx, "GET", url, nil)
	if err != nil {
		logger.Errorf("%s: forwarding to %q failed: %v", r.RemoteAddr, e.Name, err)
		http.Error(w, "internal server error", http.StatusInternalServerError)
		return
	}

	res, err := http.DefaultTransport.RoundTrip(outReq)
	if err != nil {
		logger.Errorf("%s: forwarding to %q failed: %v", r.RemoteAddr, e.Name, err)
		http.Error(w, "could not reach exporter", http.StatusBadGateway)
		return
	}
	defer res.Body.Close()

	copyHeader(w.Header(), res.Header)
	w.WriteHeader(res.StatusCode)

	if _, err := io.Copy(w, res.Body); err != nil {
		logger.Errorf("%s: copying response from %q failed: %v", r.RemoteAddr, e.Name, err)
		return
	}
}

func (e *Exporter) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodGet {
		http.Error(w, fmt.Sprintf("method %q not allowed", r.Method), http.StatusMethodNotAllowed)
		return
	}

	if e.Port != 0 {
		e.serveHTTPForward(w, r)
		return
	}

	if e.Gatherer != nil {
		h := promhttp.HandlerFor(e.Gatherer, promhttp.HandlerOpts{})
		h.ServeHTTP(w, r)
		return
	}

	w.WriteHeader(500)
	fmt.Fprintf(w, "invalid exporter configuration (no port, no gatherer)")
}

func copyHeader(dst, src http.Header) {
	for k, vv := range src {
		for _, v := range vv {
			dst.Add(k, v)
		}
	}
}

func (e *Exporter) externalPath() string {
	return "/metrics/" + e.Name
}
